#include <Region.h>
#include <cassert>

Region::RegionVector Region::getNeighboringRegions(const Point target, const coord closestDistance) const {
    const coord halfRegionSize = Parameters::getRegionSize()/2;
    // Compute coordinates of neighboring regions; some may be invalid.
    const int i = i_ + (target.getX() - halfRegionSize < i_* Parameters::getRegionSize() ?-1:1);
    const int j = j_ + (target.getY() - halfRegionSize < j_* Parameters::getRegionSize() ?-1:1);
    const bool iIsValid = i>=0 && i<Parameters::getNumRegions();
    const bool jIsValid = j>=0 && j<Parameters::getNumRegions();

    RegionVector regions;
    coord distanceX;
    coord distanceY;
    // Neighboring regions are included only if they are closer than closestDistance
    if(iIsValid) {
        const Region region(i, j_);
        distanceX = region.getClosestDistance(target);
        if(distanceX<closestDistance) regions.push_back(region);
        }
    if(jIsValid) {
        const Region region(i_, j);
        distanceY = region.getClosestDistance(target);
        if(distanceY<closestDistance) regions.push_back(region);
        }
    if(iIsValid && jIsValid && regions.size()==2) {        
        // Order closer non-diagonal region to be first
        if(distanceY<distanceX) std::swap(regions[0], regions[1]);

        // Diagonal region can be less than closestDistance away
        // only if X and Y neighboring regions both are.
        const coord distanceToDiagonalRegion = distanceX + distanceY;  // Theorem
        if(distanceToDiagonalRegion<closestDistance) regions.push_back(Region(i, j));
        }

    return regions;
}

namespace {

coord getClosestCoord(const coord regionCoord, const coord pointCoord) {
         if(pointCoord<  regionCoord     *Parameters::getRegionSize()) return regionCoord      *Parameters::getRegionSize();
    else if(pointCoord>=(regionCoord + 1)*Parameters::getRegionSize()) return (regionCoord + 1)*Parameters::getRegionSize() - 1;
    else                                                               return pointCoord;
}

}

coord Region::getClosestDistance(const Point point) const 
    {return distance(point, Point(getClosestCoord(i_, point.getX()), getClosestCoord(j_, point.getY())));}
   
// Returns the distance from target to closest point on screen that is 
// in an anti-neighboring Region.  An anti-neighboring Region is a Region
// opposite to a neighboring Region, in the sense that it's on the opposite
// side of the target's Region.
// Returned value is always between half Region size and Region size.
coord Region::getRadialDistance(const Point target) {
    const Region region(target);
    const coord halfRegionSize = Parameters::getRegionSize()/2;
    // Compute coordinates of anti-neighboring regions; some may be invalid.
    const int i = region.i_ + (target.getX() - halfRegionSize < region.i_* Parameters::getRegionSize() ?1:-1);
    const int j = region.j_ + (target.getY() - halfRegionSize < region.j_* Parameters::getRegionSize() ?1:-1);
    const bool iIsValid = i>=0 && i<Parameters::getNumRegions();
    const bool jIsValid = j>=0 && j<Parameters::getNumRegions();

    // Compute distance to anti-neighboring regions
    coord distanceX = iIsValid?Region(i, region.j_).getClosestDistance(target):Parameters::getRegionSize();
    coord distanceY = jIsValid?Region(region.i_, j).getClosestDistance(target):Parameters::getRegionSize();
    const coord result = std::min(distanceX, distanceY);
    assert(result>=halfRegionSize && result<=Parameters::getRegionSize());
    return result;
}