#include <Points.h>
#include <Region.h>
#include <SolutionConfiguration.h>
#include <algorithm>

class Points::PointReference {
    public:
        PointReference() : regionOrdinal_(0) {}
        explicit PointReference(const int* const regionOrdinal) : regionOrdinal_(regionOrdinal) {}
        const int* getRegionOrdinalPtr() const {return regionOrdinal_;}

        friend bool operator<(const PointReference pr1, const PointReference pr2)
            {return *pr1.getRegionOrdinalPtr() < *pr2.getRegionOrdinalPtr();}

    private:
        const int* regionOrdinal_;
};

Points::Points(const PointVector& inputPoints) 
    : points_(inputPoints.size()), 
      regionIterators_(Parameters::getNumRegions()*Parameters::getNumRegions() + 1) {

    // Create Region ordinal collection
    std::vector<int> regionOrdinals(inputPoints.size());
    const int offset = reinterpret_cast<const int*>(&inputPoints[0]) - &regionOrdinals[0];
    for(unsigned int k = 0; k<regionOrdinals.size(); ++k) 
        regionOrdinals[k] = Region(inputPoints[k]).getOrdinal();

    // Create reference collection
    std::vector<PointReference> pointReferences(inputPoints.size());
    for(unsigned int k = 0; k<pointReferences.size(); ++k) 
        pointReferences[k] = PointReference(&regionOrdinals[k]);

    // Sort reference collection
    std::sort(pointReferences.begin(), pointReferences.end());

    // Copy Point collection into sorted point collection,
    // and set Region iterators
    int previousOrdinal = -1;
    unsigned int regionIndex = 0;
    for(unsigned int k = 0; k<pointReferences.size(); ++k) {
        const int* const ordinalPtr = pointReferences[k].getRegionOrdinalPtr();
        points_[k] = *reinterpret_cast<const Point*>(ordinalPtr + offset);

        while(*ordinalPtr!=previousOrdinal) {  // Some regions may have no points
            ++previousOrdinal;
            regionIterators_[regionIndex++] = &points_[k];
            }
        }

    // Last regions may have no points.  Also need end iterator of last region.
    const Point* endIterator = &points_[points_.size()-1] + 1;
    while(regionIndex<regionIterators_.size()) regionIterators_[regionIndex++] = endIterator;
}

SolutionConfiguration Points::findClosest(const Point target) const {
    SolutionConfiguration solutionConfiguration(target, regionIterators_);
    const Region targetRegion(target);
    solutionConfiguration.search(targetRegion.getOrdinal(), 0);

    if(solutionConfiguration.getClosestDistance()<=1) return solutionConfiguration;

    // Search in up to three neighboring regions:
    const Region::RegionVector regions(targetRegion.getNeighboringRegions(target, 
        solutionConfiguration.getClosestDistance()));

    typedef Region::RegionVector::const_iterator const_iterator;
    for(const_iterator it = regions.begin(); it!= regions.end(); ++it) {
        const Region region = *it;

        // Test is redundant only on first iteration:
        const coord closestDistance = region.getClosestDistance(target);
        if(it!=regions.begin() && closestDistance>=solutionConfiguration.getClosestDistance()) break;

        // Search in neighboring region:
        const coord dist = Region::getRadialDistance(target);  // What to do sith this???
        solutionConfiguration.search(region.getOrdinal(), closestDistance);
        if(solutionConfiguration.getClosestDistance()==1) break;
        }

    return solutionConfiguration;
}    