#include <SolutionConfiguration.h>
#include <algorithm>

//---------------------- Helper classes for Points ------------------------

namespace {

class InsertPointFunction : public std::unary_function<const Point&, void> {
    typedef Points::PointCollection PointCollection;
    typedef PointCollection::value_type Pair;

    public:
    	InsertPointFunction(PointCollection& points) : points_(points) {} 
    	result_type operator()(argument_type point) {points_.insert(Pair(point.getKey(), point));}
                                                // Check for successful insert???
    private:
        PointCollection& points_;
};

//-------------------------------------------------------------------------

class PointIterator {
    public:
        explicit PointIterator(const Point target) 
            : target_(target), position_(target_), distance_(0), diamondIndex_(0) {}

        Point operator*() const {return position_;}

        void operator++() {
            increment();
            // Skip over positions outside screen:
            while(isInvalid(position_)) increment();
            }

    private:
        void increment();

        const Point target_;
        Point position_;  // Position of iterator
        coord distance_;  // ... from target to any point on diamond (L1 norm)
        unsigned int diamondIndex_;  // Index of position around diamond.
};

// Iterate in an expanding spiral around target
// (Actually, a series of concentric diamonds)
void PointIterator::increment() {
    if(++diamondIndex_==4*distance_ || distance_==0) {
        // Begin larger diamond, starting south of target
        diamondIndex_ = 0;
        ++distance_;
        position_ = Point(target_.getX(), target_.getY() - distance_);
        }

    // Iterate around a diamond at distance_ around target_:
    else switch((diamondIndex_-1)/distance_) {
        case 0: position_.incrementNortheast(); break;
        case 1: position_.incrementNorthwest(); break;
        case 2: position_.incrementSouthwest(); break;
        case 3: position_.incrementSoutheast(); break;
        default:  throw 9;  // Sanity check
        }        
}

}

//=========================== Points implementation =======================================

Points::Points(const PointVector& inputPoints) 
    {std::for_each(inputPoints.begin(), inputPoints.end(), InsertPointFunction(points_));}

SolutionConfiguration Points::findClosest(const Point target) const {
    SolutionConfiguration solutionConfiguration(target);
    const PointCollection::const_iterator end(points_.end());
    PointIterator it(target);
    while(end==points_.find((*it).getKey())) ++it;  // Assumes collection is nonempty
    solutionConfiguration.setClosestPoint(*it);
    return solutionConfiguration;
}    